/**********************************************************************
// Copyright 2018 Autodesk, Inc.  All rights reserved. 
**********************************************************************/

-- Conditionally loads the ALC, ALC2, ADSL and CRP scripts at the proper time in the startup sequence, registers menu item for resetting startup options
-- version 1.0 - initial version of script in package file
-- version 1.0.1 - do not display permission dialog on 3dsmax startup if loading from appstore package
-- version 1.1 - add detection and handling of ALC2

-- load structure that handles dialogs
if ::SecurityToolsDialogs_instance == undefined do
(
	local myPath = pathconfig.removePathLeaf (getThisScriptFilename())
	local pre2017 = (maxVersion())[1] < 19000
	local dialogScriptPath = myPath
	if pre2017 do 
		dialogScriptPath = pathconfig.appendPath myPath "pre2017"
	filein (pathconfig.appendPath dialogScriptPath "SecurityToolsDialogs.ms")
)
	
struct SecurityToolsStartup_structdef 
(
	private
	-- default load behavior when in quiet mode
	m_load_security_tools_quiet_mode_default = true, 
	-- whether to display listener output when not in quiet mode
	m_force_print_to_listener = false,
	-- ini file stuff
	m_ini_section = "SecurityTools",
	m_dont_show_again_ini_key = "StartupDialogDontShowAgain",
	m_load_security_tools_scripts_ini_key = "LoadSecurityTools",
	m_max_ini_file = getMAXIniFile(),
	-- my directory
	m_security_tools_scripts_dir = pathconfig.removePathLeaf (getThisScriptFilename()),
	m_security_tools_help_topic_id = "idh_security_tools",
	m_is_not_appstore_package = pathConfig.pathsResolveEquivalent (pathconfig.removePathLeaf (pathconfig.removePathLeaf (pathconfig.removePathLeaf (pathconfig.removePathLeaf m_security_tools_scripts_dir)))) @"$max\ApplicationPlugins",
	
	public
	-- A function to load the security tools scripts
	fn load_security_tools_scripts =
	(
		local log_it = false
		if ::ALC_SecurityTool == undefined do
		(
			log_it = true
			filein (pathconfig.appendPath m_security_tools_scripts_dir "ALC_SecurityTool.ms")
		)
		if ::ALC2_SecurityTool == undefined do
		(
			log_it = true
			filein (pathconfig.appendPath m_security_tools_scripts_dir "ALC2_SecurityTool.ms")
		)
		if ::CRP_SecurityTool == undefined do
		(
			log_it = true
			filein (pathconfig.appendPath m_security_tools_scripts_dir "CRP_SecurityTool.ms")
		)
		if ::ADSL_SecurityTool == undefined do
		(
			log_it = true
			filein (pathconfig.appendPath m_security_tools_scripts_dir "ADSL_SecurityTool.ms")
		)
		if log_it and securitytools != undefined do
			securitytools.LogEventById #SecurityToolsStatus #Enabled
	),

	-- A function to dsable the security tools scripts
	fn disable_security_tools_scripts =
	(
		local log_it = false
		if ::ALC_SecurityTool != undefined do
		(
			log_it = true
			::ALC_SecurityTool.disable()
		)
		if ::ALC2_SecurityTool != undefined do
		(
			log_it = true
			::ALC2_SecurityTool.disable()
		)
		if ::CRP_SecurityTool != undefined do
		(
			log_it = true
			::CRP_SecurityTool.disable()
		)
		if ::ADSL_SecurityTool != undefined do
		(
			log_it = true
			::ADSL_SecurityTool.disable()
		)
		if log_it and securitytools != undefined do
			securitytools.LogEventById #SecurityToolsStatus #Disabled
	),

	-- A function that returns true/false for a ini file key value. 
	-- If the key is present a true value is returned if the key value is anything other than 0.
	-- If the key is not present, the default value is returned.
	fn get_bool_ini_key_value filename section key default:false=
	(
		local val = getINISetting filename section key
		if val != "" then
			val != "0"
		else
			default
	),

		-- display the permissions dialog and enable or disable security tools
	fn display_permission_dialog =
	(
		local security_tools_startup_state = get_bool_ini_key_value m_max_ini_file m_ini_section m_load_security_tools_scripts_ini_key default:true
		local load_tools = SecurityToolsDialogs_instance.display_security_tools_startup_permission_dialog security_tools_startup_state m_security_tools_help_topic_id
		setINISetting m_max_ini_file m_ini_section m_dont_show_again_ini_key "1"
		setINISetting m_max_ini_file m_ini_section m_load_security_tools_scripts_ini_key (if load_tools then "1" else "0")
		if load_tools then
			load_security_tools_scripts()
		else
			disable_security_tools_scripts()
	),
	
	-- check to see whether supposed to load the security tools scripts, and do so if needed
	fn load_security_tools_scripts_check =
	(
		-- get whether to show the startup dialog 
		-- by default do not display permission dialog on 3dsmax startup if loading from appstore package, do display if loading from $max\ApplicationPlugins
		local show_startup_dialog_default = false
		local tool_load_non_quiet_mode_default = true
		if m_is_not_appstore_package do
		(
			show_startup_dialog_default = true
			tool_load_non_quiet_mode_default = false
		)
		local startup_dialog_dont_show_again_specified = get_bool_ini_key_value m_max_ini_file m_ini_section m_dont_show_again_ini_key default:(not show_startup_dialog_default)

		if startup_dialog_dont_show_again_specified then
		(
			-- Have previously shown the permisions dialog. 
			-- See if we specified that we should load the security tools scripts.
			-- Messaging is done in create handler
			if (get_bool_ini_key_value m_max_ini_file m_ini_section m_load_security_tools_scripts_ini_key default:tool_load_non_quiet_mode_default) then
			(
				load_security_tools_scripts()
			)
			else
			(
				if securitytools != undefined do
					securitytools.LogEventById #SecurityToolsStatus #Disabled
			)
		)
		else if GetQuietMode() then
		(
			-- Have not previously shown the permisions dialog. 
			-- use default behavior for quiet mode
			if m_load_security_tools_quiet_mode_default then
			(
				logsystem.logEntry ~SECURITY_TOOLS_QUIET_ENABLED~ info:true broadcast:true
				format "-- %\n" ~SECURITY_TOOLS_QUIET_ENABLED~
				load_security_tools_scripts()
			)
			else
			(
				logsystem.logEntry ~SECURITY_TOOLS_QUIET_DISABLED~ info:true broadcast:true
				format "-- %\n" ~SECURITY_TOOLS_QUIET_DISABLED~
				if securitytools != undefined do
					securitytools.LogEventById #SecurityToolsStatus #Disabled
			)
		)
		else
		(
			-- previously shown the permisions dialog, and not running in quiet mode.
			-- Display the dialog
			display_permission_dialog()
		)
	),
	
	-- Depending on when this script is run during the startup process, the dialogs may be displayed
	-- before the main 3ds Max window is displayed. This is a bit confusing and the dialogs may be hidden
	-- behind other windows. So we want to delay performing the checks that display the dialogs 
	-- until after the main 3ds Max window is shown. There's not a robust way to detect if this is the 
	-- case though prior to 3ds Max 2019.Update2

	-- The following tests the title of the 3ds Max window. If just "3dsmax" or "", then not yet displayed
	-- Title will be "" in older versions of 3ds Max (2016), "3dsmax" in newer versions
	fn max_hwnd_up_test =
	(
		local title = (windows.getHWNDData #max)[5]
		(title != "3dsmax" and title != "") -- return value
	),

	-- function to add menu item to Customize menu of main tool bar
	fn add_menu_item =
	(
		callbacks.removeScripts id:#SecurityToolsStartup_AddMenuItem
		-- make sure we find the customize menu item before trying to add our menu item to it
		local customize_Menu  = menuMan.findMenu ~SECURITY_TOOLS_MAIN_MENU_CUSTOMIZE_MENU_NAME~
		if customize_Menu != undefined do
		(
			-- create an menu item that points at our macroscript. Following must match the name and internalCategory values of Macro_SecurityToolsStartup.mcr
			local ai = menuMan.createActionItem "securityToolsStartup" "Security Tools"
			if ai != undefined do
			(
				-- only add our menu item once. This allows the user to remove the item if they so desire.
				if menuMan.registerMenuContext 0x7e25cf9 then
				(
					-- successfully created menu item, add it to the end of the Customize menu 
					customize_Menu.addItem  ai -1
					-- redraw the menu bar with the new item
					menuMan.updateMenuBar()
				)
			)
		)
	),
	
	on create do
	(
		callbacks.removeScripts id:#SecurityToolsStartup
		callbacks.removeScripts id:#SecurityToolsStartup_AddMenuItem
		::SecurityToolsStartup_instance = this
		
		-- see if 3ds Max 2019.Update2 or later
		local version = maxversion()
		local _2019_update2_or_later = version.count >= 5 and (version[4] > 21 or (version[4] == 21 and version[5] >= 2))

		-- set up to add a "3ds Max Security Tools" menu item to the Customize menu
		callbacks.addScript #postSystemStartup "::SecurityToolsStartup_instance.add_menu_item()" id:#SecurityToolsStartup_AddMenuItem
			
		local ok_to_load_security_tools = true
			
		-- by default do not display permission dialog on 3dsmax startup if loading from appstore package, do display if loading from $max\ApplicationPlugins
		local show_startup_dialog_default = false
		local tool_load_non_quiet_mode_default = true
		if m_is_not_appstore_package do
		(
			show_startup_dialog_default = true
			tool_load_non_quiet_mode_default = false
		)
		
		-- can force security tools load / no load via ini file settings
		local startup_dialog_dont_show_again_specified = get_bool_ini_key_value m_max_ini_file m_ini_section m_dont_show_again_ini_key default:(not show_startup_dialog_default)
		if startup_dialog_dont_show_again_specified then
		(
			-- Have previously shown the permisions dialog. 
			-- See if we specified that we should not load the security tools scripts.
			if (get_bool_ini_key_value m_max_ini_file m_ini_section m_load_security_tools_scripts_ini_key default:tool_load_non_quiet_mode_default) then
			(
				logsystem.logEntry ~SECURITY_TOOLS_INI_ENABLED~ info:true broadcast:true
				if GetQuietMode() or m_force_print_to_listener do
					format "-- %\n" ~SECURITY_TOOLS_INI_ENABLED~
			)
			else
			(
				ok_to_load_security_tools = false
				logsystem.logEntry ~SECURITY_TOOLS_INI_DISABLED~ info:true broadcast:true
				if GetQuietMode() or m_force_print_to_listener do
					format "-- %\n" ~SECURITY_TOOLS_INI_DISABLED~
			)
		)
		else 
		(
			-- Have not previously shown the permisions dialog. 
			-- if quiet mode, use quiet mode default setting
			if (GetQuietMode()) then
			(
				ok_to_load_security_tools = m_load_security_tools_quiet_mode_default
			)
		)

		if ok_to_load_security_tools then
		(
			-- determine if we should check to see if should load the security tools scripts now or wait until later
			local perform_load_security_tools_scripts_check_now
			if (not GetQuietMode()) then
			(
				-- in 3ds Max 2019.Update2, can tell the mxs startup state using method getMaxscriptStartupState
				if _2019_update2_or_later then
					perform_load_security_tools_scripts_check_now = getMaxscriptStartupState #stdScriptsLoaded
				else
					perform_load_security_tools_scripts_check_now = max_hwnd_up_test() 
			)
			else
			(
				-- if in quiet mode, want to check to see if should load the security tools scripts now. If in quiet mode
				-- all the behavior will be specified by default behaviors and no dialogs will be displayed.
				perform_load_security_tools_scripts_check_now = true
			)
			
			if perform_load_security_tools_scripts_check_now then
				load_security_tools_scripts_check()
			else
			(
				-- the #preStartupScriptLoad event is new to 3ds Max 2019.Update2, and fires immediately before loading the startup scripts
				if _2019_update2_or_later then
					callbacks.addScript #preStartupScriptLoad "::SecurityToolsStartup_instance.load_security_tools_scripts_check()" id:#SecurityToolsStartup
				else
					callbacks.addScript #postSystemStartup "::SecurityToolsStartup_instance.load_security_tools_scripts_check()" id:#SecurityToolsStartup
			)
		)
		else
		(
			if securitytools != undefined do
				securitytools.LogEventById #SecurityToolsStatus #Disabled
		)
	)
)

(
	SecurityToolsStartup_structdef()
	silentValue
)

-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQB92ZZDJiZjQVRq6eMq5poPdYbnn3kS/pXC
-- YgJnCM960+rP8bCftiXtM2QypjBxuGVe2jQYqLqFaDY2CabV2i8fhfOypKn/PayU
-- fM4EdDd8SfZ9LIP43sbW/FFwFebGY01QUhktXIZ+l7oHraepRfvrV19P5JMfrOt3
-- Gii5EEusqV6xTgz31FI2t1tMAw0Thsx2ctPMHXOjR1YnnvPUr7NSnWrvzoC9Vb2O
-- Da1XN4bw/F5mg/AfY2mXJrXDPmxcLnhK2PD4bVWarK0A17+WXyacyl2JSJ4Y2mNL
-- xxOX5qtQ/8d6yyKGtifDYYl4U3J32n0wUA/DHKPN6bw0645BONe/
-- -----END-SIGNATURE-----